//
//	Copyright  Nemetschek Vectorworks, Inc.
//	Use of this file is governed by the Nemetschek Vectorworks SDK License Agreement
//	http://developer.vectorworks.net/index.php?title=Vectorworks_SDK_License
//
// This file contains the necessary VCOM interfaces required for reading and writing workspace files for VectorWorks version 13.
 

#ifndef VCOM_INTERFACE_WORKSPACES__H
#define VCOM_INTERFACE_WORKSPACES__H

#include "../UI/IIcon.h"
#include "../Filing/IXMLFile.h"

#if GS_MAC
	#define NNA_DEPRECATED __attribute__ ((deprecated))
#else
	#define NNA_DEPRECATED
#endif

#pragma once

/**
	@brief A interface to a workspace-file tool.
*/

namespace VectorWorks
{
	namespace Workspaces
	{
	
		enum EWorkspacesError {
			eWorkspacesError_NoError,
			eWorkspacesError_UnknownError,
			eWorkspacesError_InvalidArg,
			eWorkspacesError_WorkspaceNotOpen,
			eWorkspacesError_WorkspaceAlreadyOpen,
			eWorkspacesError_WorkspaceAlreadyExists,
			eWorkspacesError_WorkspaceNotFound,
			eWorkspacesError_WorkspaceNotReadable,
			eWorkspacesError_WorkspaceNotReadWritable,
			eWorkspacesError_BadArrayIndex,
			eWorkspacesError_ElementNotFound,
			eWorkspacesError_TooManySpecialMenuItems
		};

		const Sint32 kWorkspaceVersion12		= 2;
		const Sint32 kWorkspaceVersion13		= 3;
		const Sint32 kWorkspaceVersion14		= 3; //!< no change.
		const Sint32 kWorkspaceVersion15		= 4; //!< removed the working planes palette from the standard palettes chunk
		const Sint32 kWorkspaceVersionCurrent = kWorkspaceVersion15;
		
		/**
			@class IWorkspaceTool
		*/
		class DYNAMIC_ATTRIBUTE IWorkspaceTool : public IVWUnknown
		{
		public:
			virtual ~IWorkspaceTool() {}
			
			/**
				@brief Gets the workspace identifier for the menu item.
				@param outIdentifier identifier in use for this menu item.
				@return kVCOMError_NoError if the operation was successful.

				@author Lyndsey Ferguson
			 */
			virtual VCOMError VCOM_CALLTYPE GetIdentifier(TXString& outIdentifier) = 0;
			
			/**
				@brief Sets the universal name for the tool.
				@param inName the universal name for this tool.
				@return kVCOMError_NoError if the operation was successful.

				@author Lyndsey Ferguson
			 */
			virtual VCOMError VCOM_CALLTYPE SetUniversalName(const TXString& inName) = 0;

			/**
				@brief Gets the universal name for the tool.
				@param outName on output, if successful, the universal name for this tool.
				@return kVCOMError_NoError if the operation was successful.

				@author Lyndsey Ferguson
			 */
			virtual VCOMError VCOM_CALLTYPE GetUniversalName(TXString& outName) = 0;
			
			/**
				@brief Sets the Resource Manager ID for this item.
				@param inResourceID the Resource Manager ID to save.
				@return kVCOMError_NoError if the operation was successful.

				@author Lyndsey Ferguson
			 */
			virtual VCOMError VCOM_CALLTYPE SetResourceManagerID(Sint16 inResourceID) = 0;
			
			/**
				@brief Gets the Resource Manager ID if there is one.
				@param outResourceManagerID is the resource manager file id.
				@return kVCOMError_NoError if the menu has a Resource Manager ID.

				@author Lyndsey Ferguson
			*/
			virtual VCOMError VCOM_CALLTYPE GetResourceManagerID(Sint16& outResourceManagerID) = 0;
				
			/**
				@brief Sets the shortcut key information.
				@param inShortcutKey the shortcut key character.
				@param inKeyModifiers the shortcut key modifiers.
				@return kVCOMError_NoError if the operation was successful.

				@author Lyndsey Ferguson
				@date 10/02/2006
			 */
			virtual VCOMError VCOM_CALLTYPE SetShortcutKeyInfo(char inShortcutKey, Uint32 inKeyModifiers) = 0;
			
			/**
				@brief Delete shortcut key.
				@return kVCOMError_NoError if the operation was successful.
			 */
			virtual VCOMError VCOM_CALLTYPE DeleteShortcutKey() = 0;
			
			/**
				@brief Gets the shortcut key information.
				@param outShortcutKey on output, if successful, the shortcut key character.
				@param outKeyModifiers on output, if successful, the shortcut key modifiers.
				@return kVCOMError_NoError if the operation was successful.

				@author Lyndsey Ferguson
				@date 10/02/2006
			 */
			virtual VCOMError VCOM_CALLTYPE GetShortcutKeyInfo(char& outShortcutKey, Uint32& outKeyModifiers) = 0;
			/**
				@brief Gets the last error that was generated by the IWorkspaces interface.
				@param outError is the last error that occurred. Any successful call to a IWorkspaces function will reset this to EWorkspaces_NoError.
			 
				@author Lyndsey Ferguson
			 */
			virtual VCOMError VCOM_CALLTYPE GetLastError(EWorkspacesError& outError) = 0;
	
		};

		/**
			@class IWorkspaceIconProvider
		 
			This class is the basis for various icon formats.
		*/
		class DYNAMIC_ATTRIBUTE IWorkspaceIconProvider : public IVWUnknown
		{
		public:
			virtual ~IWorkspaceIconProvider() {}
		};
		
		/**
			@class IWorkspaceToolSet
		*/


		class DYNAMIC_ATTRIBUTE IWorkspaceToolSet : public IVWUnknown
		{
		public:
			virtual ~IWorkspaceToolSet() {}
			
			/**
				@brief Gets the workspace identifier for the menu item.
				@param outIdentifier identifier in use for this menu item.
				@return kVCOMError_NoError if the operation was successful.

				@author Lyndsey Ferguson
			 */
			virtual VCOMError VCOM_CALLTYPE GetIdentifier(TXString& outIdentifier) = 0;

			/**
				@brief Sets the display name for the tool palette.
				@param inName is the display name that this tool palette should use.
				@return kVCOMError_NoError if the operation was successful.

				@author Lyndsey Ferguson
			 */
			virtual VCOMError VCOM_CALLTYPE SetDisplayName(const TXString& inName) = 0;
			
			/**
				@brief Gets the display name for the tool set.
				@param outName on output, if successful, the display name for this tool set.
				@return kVCOMError_NoError if the operation was successful.

				@author Lyndsey Ferguson
			 */
			virtual VCOMError VCOM_CALLTYPE GetDisplayName(TXString& outName) = 0;
			/**
				@brief Sets the tool set icon.
				@param inIconData a memory pointer which contains data necessary to draw an Icon.
				@param inIconDataSize the size of memory allocated for the icon.
				@return kVCOMError_NoError if the operation was successful.

				@author Lyndsey Ferguson
				@date 10/02/2006
			 */
			virtual VCOMError VCOM_CALLTYPE SetIcon(Uint8* inIconData, size_t inIconDataSize, Uint8* inIconDataRetina, size_t inIconDataSizeRetina) = 0;
			/**
				@brief Gets the tool set icon.
				@param inIconData on output, if successful, a memory pointer which contains data necessary to draw an Icon. Client must GS_Dispose this memory.
				@param inIconDataSize on output, if successful, the size of memory allocated for the icon.
			 	@param inIconDataRetina on output, if successful, a memory pointer which contains data necessary to draw a retina Icon. Client must GS_Dispose this memory.
			 	@param inIconDataSizeRetina on output, if successful, the size of memory allocated for the retina icon.
				@return kVCOMError_NoError if the operation was successful.

				@author Lyndsey Ferguson
				@date 10/02/2006
			 */
			virtual VCOMError VCOM_CALLTYPE GetIcon(Uint8*& inIconData, size_t& inIconDataSize, Uint8*& inIconDataRetina, size_t& inIconDataSizeRetina) = 0;
			/**
				@brief Adds a new tool to this tool set.
				@param inTool the tool to add.
				@return kVCOMError_NoError
			 
				@author Lyndsey Ferguson
			*/
			virtual VCOMError VCOM_CALLTYPE AddTool(IWorkspaceTool* inTool) = 0;
			/**
				@brief Adds a Sub-Tool to the given tool in the tool set.
				@param inParentTool the tool that has been added as a tool button for this tool set.
				@param inChildTool the sub-tool that should be added to the list of sub-tools for this tool button.
				@return kVCOMError_NoError if the operation was successful.
			 
				@author Lyndsey Ferguson
			*/
			virtual VCOMError VCOM_CALLTYPE AddChildTool(IWorkspaceTool* inParentTool, IWorkspaceTool* inChildTool) = 0;
			
			/**
				@brief Gets the number of tool buttons for this tool set.
				@param outToolCount on output, if successful, the count of tools in this tool set.
				@return kVCOMError_NoError if the operation was successful.
			 
				@author Lyndsey Ferguson
			*/
			virtual VCOMError VCOM_CALLTYPE GetToolCount(Sint16& outToolCount) = 0;
			/**
				@brief Gets the tool at the given index.
				@param inIndex the index into the list of buttons to retrieve the tool for.
				@param outTool on output, if successful, the tool stored at the given index.
				@param outHasSubTools on output, if successful, a bool that indicates whether or not this tool button has sub-tools.
				@return kVCOMError_NoError if the operation was successful.
			 
				@author Lyndsey Ferguson
			*/
			virtual VCOMError VCOM_CALLTYPE GetToolAtIndex(Sint16 inIndex, IWorkspaceTool** outTool, bool& outHasSubTools) = 0;
			/**
				@brief Gets the number of sub-tools for the given tool.
				@param inParentTool the tool that will be the tool button for this tool set.
				@param outSubToolCount on output, if successful, the number of sub-tools in the parent tool.
				@return kVCOMError_NoError if the operation was successful.
			 
				@author Lyndsey Ferguson
			*/
			virtual VCOMError VCOM_CALLTYPE GetSubToolCountForTool(IWorkspaceTool* inParentTool, Sint16& outSubToolCount) = 0;
			/**
				@brief Gets the sub-tool for the given parent tool at the given index.
				@param inParentTool the tool that contains all the other sub-tools.
				@param outChildTool on output, if successful, the child tool at the given index for the given tool.
				@param inIndex the index into the list of sub-tools that we want the tool for.
				@return kVCOMError_NoError if the operation was successful.
			 
				@author Lyndsey Ferguson
			 */
			virtual VCOMError VCOM_CALLTYPE GetSubToolForToolAtIndex(IWorkspaceTool* inParentTool, Sint16 inIndex, IWorkspaceTool** outChildTool) = 0;
			/**
				@brief Remove the tool at the given index.
				@param inIndex the index into the list of buttons to retrieve the tool for.
				@return kVCOMError_NoError if the operation was successful.
			 
				@author Vladislav Stanev
			*/
			virtual VCOMError VCOM_CALLTYPE RemoveToolAtIndex(Sint16 inIndex) = 0;
			/**
				@brief Gets the last error that was generated by the IWorkspaces interface.
				@param outError is the last error that occurred. Any successful call to a IWorkspaces function will reset this to EWorkspaces_NoError.
			 
				@author Lyndsey Ferguson
			 */
			virtual VCOMError VCOM_CALLTYPE GetLastError(EWorkspacesError& outError) = 0;
	
		};

		/**
			@enum EWindowCorner is the corner from where a Window's offset is calculated from.
			
			For example, if a Window's home corner is the Upper Right, and the Window has an offset of
			(10, 20), the Window is offset 10 pixels to the left of the right hand side of the display and
			20 pixels down from the top side of the display.
		 
		 */
		enum EWindowHomeCorner {
			eWindowHomeCorner_UpperLeft,
			eWindowHomeCorner_UpperRight,
			eWindowHomeCorner_LowerRight,
			eWindowHomeCorner_LowerLeft,
			eWindowHomeCorner_ScreenUpperLeft
		};

		/**
			@class IWorkspacePalette
		*/
		class DYNAMIC_ATTRIBUTE IWorkspacePalette : public IVWUnknown
		{
		public:
			virtual ~IWorkspacePalette() {}
			
			/**
				@brief Gets the workspace identifier for the menu item.
				@param outIdentifier identifier in use for this menu item.
				@return kVCOMError_NoError if the operation was successful.

				@author Lyndsey Ferguson
			 */
			virtual VCOMError VCOM_CALLTYPE GetIdentifier(TXString& outIdentifier) = 0;
			
			/**
				@brief Sets the palette's home corner.
				@param inHomeCorner one of the enumerated palette corners.
				@return kVCOMError_NoError if the operation was successful.
			 
				@see EWindowHomeCorner
				@author Lyndsey Ferguson
			*/
			virtual VCOMError VCOM_CALLTYPE SetHomeCorner(EWindowHomeCorner inHomeCorner) = 0;
			/**
				@brief Gets the palette's home corner.
				@param outHomeCorner on output, if successful, one of the enumerated palette corners.
				@return kVCOMError_NoError if the operation was successful.

				@see EWindowHomeCorner
				@author Lyndsey Ferguson
			*/
			virtual VCOMError VCOM_CALLTYPE GetHomeCorner(EWindowHomeCorner& outHomeCorner) = 0;
			/**
				@brief Sets the offset from the palette's home corner.
				@param inHorizontalOffset the horizontal offset from the palette's home corner.
				@param inVerticalOffset the vertical offset from the palette's home corner.
				@return kVCOMError_NoError if the operation was successful.
			 
				The offset is how far away from the home corner is the palette. For example, if the
				home corner is the lower-right hand side of the display, then a vertical offset of 10
				signifies that the palette is 10 units away from the bottom of the display.
			 
				@author Lyndsey Ferguson
			*/
			virtual VCOMError VCOM_CALLTYPE SetHomeCornerOffset(Sint16 inHorizontalOffset, Sint16 inVerticalOffset) = 0;
			/**
				@brief Sets the offset from the palette's home corner.
				@param outHorizontalOffset on output, if successful, the horizontal offset from the palette's home corner.
				@param outVerticalOffset on output, if successful, the vertical offset from the palette's home corner.
				@return kVCOMError_NoError if the operation was successful.

				The offset is how far away from the home corner is the palette. For example, if the
				home corner is the lower-right hand side of the display, then a vertical offset of 10
				signifies that the palette is 10 units away from the bottom of the display.

				@author Lyndsey Ferguson
			 */
			virtual VCOMError VCOM_CALLTYPE GetHomeCornerOffset(Sint16& outHorizontalOffset, Sint16& outVerticalOffset) = 0;
			/**
				@brief Sets the window shade state for the given palette.
				@param inWindowShadeState true if palette should be collapsed to its title bar, false otherwise.
				@return kVCOMError_NoError if the operation was successful.

				@author Lyndsey Ferguson
				@date 10/02/2006
			 */
			virtual VCOMError VCOM_CALLTYPE SetWindowShadeState(bool inWindowShadeState) = 0;
			/**
				@brief Gets the Window Shade state for the palette.
				@param outWindowShadeState on output, if successful, a bool which if true, means that the palette should be window shaded.
				@return kVCOMError_NoError if the operation was successful.

				@author Lyndsey Ferguson
				@date 10/02/2006
			 */
			virtual VCOMError VCOM_CALLTYPE GetWindowShadeState(bool& outWindowShadeState) = 0;
			/**
				@brief Gets the visibility state for the palette.
				@param inVisibilityeState true if this palette should be visible, false otherwise.
				@return kVCOMError_NoError if the operation was successful.

				@author Lyndsey Ferguson
				@date 10/02/2006
			 */
			virtual VCOMError VCOM_CALLTYPE SetVisibilityState(bool inVisibilityeState) = 0;
			/**
				@brief Gets the visibility state of the palette.
				@param outVisibilityeState on output, if successful, a bool that indicates whether or not the palette is visible.
				@return kVCOMError_NoError if the operation was successful.

				@author Lyndsey Ferguson
				@date 10/02/2006
			 */
			virtual VCOMError VCOM_CALLTYPE GetVisibilityState(bool& outVisibilityeState) = 0;
			/**
				@brief Gets the last error that was generated by the IWorkspaces interface.
				@param outError is the last error that occurred. Any successful call to a IWorkspaces function will reset this to EWorkspaces_NoError.
			 
				@author Lyndsey Ferguson
			 */
			virtual VCOMError VCOM_CALLTYPE GetLastError(EWorkspacesError& outError) = 0;
	
		};

		/**
			@class IWorkspaceResizablePalette
		 
			@brief This class handles the workspace settings for resizable palettes.
		*/
		class IWorkspaceSizeablePalette : public IWorkspacePalette
		{
		public:
			virtual ~IWorkspaceSizeablePalette() {}

			/**
				@brief Sets the width and height for the palette.
				@param inWidth the width to set for this palette.
				@param inHeight the height to set for this palette.
				@return kVCOMError_NoError if the operation was successful.
			 
				@author Lyndsey Ferguson
			*/
			virtual VCOMError VCOM_CALLTYPE SetDimensions(Sint16 inWidth, Sint16 inHeight) = 0;
			
			/**
				@brief Gets the width and height for the palette.
				@param outWidth the current width of the palette.
				@param outHeight the current height of the palette.
				@return kVCOMError_NoError if the operation was successful.
				
				@author Lyndsey Ferguson
			*/
			virtual VCOMError VCOM_CALLTYPE GetDimensions(Sint16& outWidth, Sint16& outHeight) = 0;
		};
		
		
		/**
			@enum EResourceDisplayStyles
			This enumeration designates the different styles that the resource browser can display its resources in.
		*/
		enum EResourceDisplayStyles {
			eResourceDisplayStyle_List,
			eResourceDisplayStyle_Thumbnail
		};
		
		/**
			@enum EResourceGroups
			This enumeration designates the different resource types that the resource browser deals with.
		*/
		enum EResourceGroups {
			eResourceGroup_Plugins,
			eResourceGroup_Hatches,
			eResourceGroup_RecordFormats,
			eResourceGroup_VectorScriptPaletes,
			eResourceGroup_Worksheets,
			eResourceGroup_Textures,
			eResourceGroup_SymbolFolders,
			eResourceGroup_RenderBackgrounds,
			eResourceGroup_Images,
			eResourceGroup_Gradients,
			eResourceGroup_SketchStyles,
			eResourceGroup_WallStyles,
			eResourceGroup_SlabStyles,
			eResourceGroup_Tiles,
			eResourceGroup_TextStyles,
            eResourceGroup_LineTypes,
            eResourceGroup_RoofStyles
		};
		const Sint16 kMaxNumberOfResourceGroups = 17; // This the number of enumerations there are in the enum EResourceGroups.
		
		/**
			@class IWorkspaceAttributesPalette
		*/
		class IWorkspaceAttributesPalette : public IWorkspaceSizeablePalette
		{
		public:
			virtual ~IWorkspaceAttributesPalette() {}

			/**
				@brief Sets the Zoom Expansion State for the Attribute palette.
				@param inZoomed true if the palette should be zoom/expanded, false otherwise.
				@return kVCOMError_NoError if the operation was successful.
			 
				@author Lyndsey Ferguson
			*/
			virtual VCOMError VCOM_CALLTYPE SetZoomState(bool inZoomed) = 0;
			
			/**
				@brief Gets the Zoom Expansion State for the Attribute palette.
				@param outZoomed true if the palette should be zoom/expanded, false otherwise.
				@return kVCOMError_NoError if the operation was successful.

				@author Lyndsey Ferguson
			*/
			virtual VCOMError VCOM_CALLTYPE GetZoomState(bool& outZoomed) = 0;
		};
		
		/**
			@class IWorkspaceObjectInfoPalette
		*/
		class IWorkspaceObjectInfoPalette : public IWorkspaceSizeablePalette
		{
		public:
			virtual ~IWorkspaceObjectInfoPalette() {}
			
			/**
				@brief Sets the offset of the separators that lie between the various panes in the Data panel (idx = 1) or Render panel (idx = 2,3) for this palette.
				@param inSeparatorOffsetOne the offset for the first separator.
				@return kVCOMError_NoError if the operation was successful.
			 
				@author Lyndsey Ferguson
			*/
			virtual VCOMError VCOM_CALLTYPE SetSeparatorOffsets(Sint16 inSeparatorOffsetOne, short idx) = 0;
			
			/**
				@brief Gets the offset of the separators that lie between the various panes in the Data panel (idx = 1) or Render panel (idx = 2,3) for this palette.
				@param outSeparatorOffsetOne the offset for the first separator.
				@return kVCOMError_NoError if the operation was successful.

				@author Lyndsey Ferguson
			 */
			virtual VCOMError VCOM_CALLTYPE GetSeparatorOffsets(Sint16& outSeparatorOffsetOne, short idx) = 0;
			
			/**
				@brief Sets the active panel index (Shape=0, Data=1, Render=2)
				@param inPanelIndex the index of the panel that is to be saved as active.
				@return kVCOMError_NoError if the operation was successful.
			 
				@author Lyndsey Ferguson
			*/
			virtual VCOMError VCOM_CALLTYPE SetActivePanel(Sint16 inPanelIndex) = 0;
			
			/**
				@brief Gets the active panel index (Shape=0, Data=1, Render=2)
				@param outPanelIndex the index of the panel that was saved as active.
				@return kVCOMError_NoError if the operation was successful.

				@author Lyndsey Ferguson
			*/
			virtual VCOMError VCOM_CALLTYPE GetActivePanel(Sint16& outPanelIndex) = 0;
			
			/**
			 @brief Set hosting palette ID.
			 @param paletteID The hosting palette ID of this pane.
			 @return VCOM error code.
			 */
			virtual VCOMError VCOM_CALLTYPE SetHostingPaletteID(Sint16 paletteID) = 0;
			
			/**
			 @brief Get hosting palette ID.
			 @param paletteID The hosting palette ID of this pane.
			 @return VCOM error code.
			 */
			virtual VCOMError VCOM_CALLTYPE GetHostingPaletteID(Sint16& paletteID) = 0;
		};
		
		/**
			@typedef IWorkspaceMessagePalette
			@see IWorkspaceSizeablePalette
		*/
		typedef IWorkspaceSizeablePalette IWorkspaceMessagePalette;
		
		/**
			@class IWorkspaceNavigationPalette
		*/
		class IWorkspaceNavigationPalette : public IWorkspaceSizeablePalette
		{
		public:
			virtual ~IWorkspaceNavigationPalette() {}
			
			/**
				@brief Sets the active panel index (Class=0, Layer=1, Sheet Layer=2, Viewport=3, Saved View=4)
				@param inPanelIndex the index of the panel that is to be saved as active.
				@return kVCOMError_NoError if the operation was successful.

				@author Lyndsey Ferguson
			 */
			virtual VCOMError VCOM_CALLTYPE SetActivePanel(Sint16 inPanelIndex) = 0;
			
			/**
				@brief Gets the active panel index (Class=0, Layer=1, Sheet Layer=2, Viewport=3, Saved View=4)
				@param outPanelIndex the index of the panel that was saved as active.
				@return kVCOMError_NoError if the operation was successful.

				@author Lyndsey Ferguson
			 */
			virtual VCOMError VCOM_CALLTYPE GetActivePanel(Sint16& outPanelIndex) = 0;
			
			/**
			 @brief Set hosting palette ID.
			 @param paletteID The hosting palette ID of this pane.
			 @return VCOM error code.
			 */
			virtual VCOMError VCOM_CALLTYPE SetHostingPaletteID(Sint16 paletteID) = 0;
			
			/**
			 @brief Get hosting palette ID.
			 @param paletteID The hosting palette ID of this pane.
			 @return VCOM error code.
			 */
			virtual VCOMError VCOM_CALLTYPE GetHostingPaletteID(Sint16& paletteID) = 0;
		};

		/**
			@class IWorkspaceVisualizationPalette
		*/
		class IWorkspaceVisualizationPalette : public IWorkspaceSizeablePalette
		{
		public:
			virtual ~IWorkspaceVisualizationPalette() {}
			
			/**
				@brief Sets the active panel index (Light=0, Camera=1)
				@param inPanelIndex the index of the panel that is to be saved as active.
				@return kVCOMError_NoError if the operation was successful.
			 */
			virtual VCOMError VCOM_CALLTYPE SetActivePanel(Sint16 inPanelIndex) = 0;
			
			/**
				@brief Gets the active panel index (Light=0, Camera=1)
				@param outPanelIndex the index of the panel that was saved as active.
				@return kVCOMError_NoError if the operation was successful.
			 */
			virtual VCOMError VCOM_CALLTYPE GetActivePanel(Sint16& outPanelIndex) = 0;
			
			/**
			 @brief Set hosting palette ID.
			 @param paletteID The hosting palette ID of this pane.
			 @return VCOM error code.
			 */
			virtual VCOMError VCOM_CALLTYPE SetHostingPaletteID(Sint16 paletteID) = 0;
			
			/**
			 @brief Get hosting palette ID.
			 @param paletteID The hosting palette ID of this pane.
			 @return VCOM error code.
			 */
			virtual VCOMError VCOM_CALLTYPE GetHostingPaletteID(Sint16& paletteID) = 0;
		};
				
		class IWorkspaceResourceManagerPalette : public IWorkspaceSizeablePalette
		{
		public:
			virtual ~IWorkspaceResourceManagerPalette() {}

			virtual VCOMError VCOM_CALLTYPE SetResourceViewMode(Uint8) = 0;
			virtual VCOMError VCOM_CALLTYPE GetResourceViewMode(Uint8&) = 0;

			virtual VCOMError VCOM_CALLTYPE SetContentViewVisible(bool) = 0;
			virtual VCOMError VCOM_CALLTYPE IsContentViewVisible(bool&) = 0;

			virtual VCOMError VCOM_CALLTYPE SetContentViewWidth(Uint16) = 0;
			virtual VCOMError VCOM_CALLTYPE GetContentViewWidth(Uint16&) = 0;

			virtual VCOMError VCOM_CALLTYPE SetDetailViewVisible(bool) = 0;
			virtual VCOMError VCOM_CALLTYPE IsDetailViewVisible(bool&) = 0;

			virtual VCOMError VCOM_CALLTYPE SetDetailViewWidth(Uint16) = 0;
			virtual VCOMError VCOM_CALLTYPE GetDetailViewWidth(Uint16&) = 0;

			virtual VCOMError VCOM_CALLTYPE SetPreviewViewHeight(Uint16) = 0;
			virtual VCOMError VCOM_CALLTYPE GetPreviewViewHeight(Uint16&) = 0;
		};

		
		/**
			@typedef IWorkspaceWorkingPlanePalette
			@see IWorkspacePalette
		 */
		typedef IWorkspaceSizeablePalette IWorkspaceWorkingPlanePalette;
		/**
			@class IWorkspaceToolPalette
		 */
		
		class IWorkspaceToolPalette : public IWorkspaceSizeablePalette
		{
		public:
			virtual ~IWorkspaceToolPalette() {}
			
			/**
				@brief Sets the display name for the tool palette.
				@param inName is the display name that this tool palette should use.
				@return kVCOMError_NoError if the operation was successful.
			 
				@author Lyndsey Ferguson
			*/
			virtual VCOMError VCOM_CALLTYPE SetDisplayName(const TXString& inName) = 0;
			
			/**
				@brief Gets the display name for the tool palette.
				@param outName on output, if successful, the display name for this tool palette.
				@return kVCOMError_NoError if the operation was successful.

				@author Lyndsey Ferguson
			 */
			virtual VCOMError VCOM_CALLTYPE GetDisplayName(TXString& outName) = 0;
			
			/**
				@brief Adds a tool set to this tool palette.
				@param inToolSet the tool set that should be added to this tool palette.
				@return kVCOMError_NoError if the operation was successful.

				@author Lyndsey Ferguson
			 */
			virtual VCOMError VCOM_CALLTYPE AddToolSet(IWorkspaceToolSet* inToolSet) = 0;
			/**
				@brief Get the number of tool sets that are in the tool palette.
				@param outToolSetCount on output, if successful, the count of tool sets in the tool palette.
				@return the number of tool sets in this tool palette.

				@author Lyndsey Ferguson
			 */
			virtual VCOMError VCOM_CALLTYPE GetToolSetCount(Sint16& outToolSetCount) = 0;
			/**
				@brief Gets the tool set at the given index.
				@param inIndex the index into the array of tool sets to get.
				@param outToolSet on output, if successful, the tool set at the given index.
				@return kVCOMError_NoError if the operation was successful.

				@author Lyndsey Ferguson
			 */
			virtual VCOMError VCOM_CALLTYPE GetToolSetAtIndex(Sint16 inIndex, IWorkspaceToolSet** outToolSet) = 0;

			/**
				@brief Sets the view style for the tool sets (icon, icon & text, text)
				@param inViewStyle the view style to use for the tool sets.
				@return kVCOMError_NoError if the operation was successful.

				@author Lyndsey Ferguson
			 */
			virtual VCOMError VCOM_CALLTYPE SetToolSetViewStyle(Sint16 inViewStyle) = 0;
			
			/**
				@brief Gets the view style for the tool sets (icon, icon & text, text)
				@param outViewStyle  on output, if successful, the value representing the enumerated value in the view style enumeration.
				@return kVCOMError_NoError if the operation was successful

				@author Lyndsey Ferguson
			 */
			virtual VCOMError VCOM_CALLTYPE GetToolSetViewStyle(Sint16& outViewStyle) = 0;
			
			/**
				@brief Sets the sort style for the tool sets (ascending, descending, or manual placement)
				@param inSortStyle the sort style to use for the tool sets.
				@return kVCOMError_NoError if the operation was successful.

				@author Lyndsey Ferguson
			 */
			virtual VCOMError VCOM_CALLTYPE SetToolSetSortStyle(Sint16 inSortStyle) = 0;
			
			/**
				@brief Gets the current sort style used for the tools.
				@param outSortStyle on output, if successful, the sort style enumeration.
				@return kVCOMError_NoError if the operation was successful.

				@author Lyndsey Ferguson
			 */
			virtual VCOMError VCOM_CALLTYPE GetToolSetSortStyle(Sint16& outSortStyle) = 0;
			/**
				@brief Sets the view style for the tool sets (icon, icon & text, text)
				@param inViewStyle the view style to use for the tool sets.
				@return kVCOMError_NoError if the operation was successful.

				@author Lyndsey Ferguson
			 */
			virtual VCOMError VCOM_CALLTYPE SetToolViewStyle(Sint16 inViewStyle) = 0;
			
			/**
				@brief Gets the view style for the tools (icon, icon & text, text)
				@return the number representing the enumeration of the view style enumeration.

				@author Lyndsey Ferguson
			 */
			virtual VCOMError VCOM_CALLTYPE GetToolViewStyle(Sint16& outViewStyle) = 0;
			/**
				@brief Sets the sort style for the tools (ascending, descending, or manual placement)
				@param inSortStyle the sort style to use for the tools.
				@return kVCOMError_NoError if the operation was successful.

				@author Lyndsey Ferguson
			 */
			virtual VCOMError VCOM_CALLTYPE SetToolSortStyle(Sint16 inSortStyle) = 0;
			
			/**
				@brief Gets the current sort style used for the tools.
				@param outSortStyle on output, if successful, the sort style enumeration.
				@return kVCOMError_NoError if the operation was successful.

				@author Lyndsey Ferguson
			 */
			virtual VCOMError VCOM_CALLTYPE GetToolSortStyle(Sint16& outSortStyle) = 0;
			/**
				@brief The position of the group of tool sets in the tool palette.
				@param inTopPosition true if we want this palette to display its group of tool sets at the top of the palette. False if it should be at the bottom.
				@return kVCOMError_NoError if the operation was successful.

				@author Lyndsey Ferguson
				@date 10/02/2006
			 */
			virtual VCOMError VCOM_CALLTYPE SetToolSetPosition(bool inTopPosition) = 0;
			/**
				@brief The position of the group of tool sets in the tool palette.
				@param outToolSetPosition on output, if successful, true if the tool sets should be grouped at the top of the tool palette, false for the bottom of the tool palette.
				@return kVCOMError_NoError if the operation was successful.

				@author Lyndsey Ferguson
				@date 10/02/2006
			 */
			virtual VCOMError VCOM_CALLTYPE GetToolSetPosition(bool& outToolSetPosition) = 0;
			/**
				@brief Sets the active tool set by its index.
				@param inToolSetIndex the index of the tool set that is to be active for the given tool palette.
				@return kVCOMError_NoError if the operation was successful.

				@author Lyndsey Ferguson
				@date 10/02/2006
			 */
			virtual VCOMError VCOM_CALLTYPE SetActiveToolSet(Sint16 inToolSetIndex) = 0;
			/**
				@brief Gets the active tool set for this tool palette.
				@param outActiveToolSetIndex on output, if successful, the index of the active tool set.
				@return kVCOMError_NoError if the operation was successful.

				@author Lyndsey Ferguson
				@date 10/02/2006
			 */
			virtual VCOMError VCOM_CALLTYPE GetActiveToolSet(Sint16& outActiveToolSetIndex) = 0;
			/**
				@brief Sets the status flag that indicates whether or not this is a tear-off tool palette.
				@param inTearOffState true if this is a tear-off tool palette.
				@return kVCOMError_NoError if the operation was successful.

				@author Lyndsey Ferguson
			 */
			virtual VCOMError VCOM_CALLTYPE SetTearOffState(bool inTearOffState) = 0;
			/**
				@brief Gets the status flag that indicates whether or not this is a tear-off tool palette.
				@param outTearOffState on output, if successful, a bool indicating whether this tool palette is a tear-off or not.
				@return kVCOMError_NoError if the operation was successful.

				@author Lyndsey Ferguson
			 */
			virtual VCOMError VCOM_CALLTYPE GetTearOffState(bool& outTearOffState) = 0;
			/**
				@brief Delete the tool set at the given index.
				@param inIndex the index into the array of tool sets to get.
				@return kVCOMError_NoError if the operation was successful.

				@author Vladislav Stanev
			 */
			virtual VCOMError VCOM_CALLTYPE DeleteToolSetAtIndex(Sint16 inIndex) = 0;
			/**
				@brief Gets the last error that was generated by the IWorkspaces interface.
				@param outError is the last error that occurred. Any successful call to a IWorkspaces function will reset this to EWorkspaces_NoError.

				@author Lyndsey Ferguson
			 */
			virtual VCOMError VCOM_CALLTYPE GetLastError(EWorkspacesError& outError) = 0;
		};
		
		class IWorkspaceMenu;
		
		/**
			@enum EWorkspaceMenuChunkType
		 
		 This enumeration denotes what type of menu chunk this menu item refers to.
		 */
		
		enum EWorkspaceMenuChunkType {
			eWorkspaceMenuChunkType_None,
			eWorkspaceMenuChunkType_Regular,
			eWorkspaceMenuChunkType_Dynamic,
			eWorkspaceMenuChunkType_Variable
		};
		
		/**
			@class IWorkspaceMenuItem
		
			A menu item can be one of two types:
			- A menu item with a shortcut key and modifier (empty values are possible).
			- A menu item that has a submenu
		*/

		class DYNAMIC_ATTRIBUTE IWorkspaceMenuItem : public IVWUnknown
		{
		public:
			virtual ~IWorkspaceMenuItem() {}
			
			/**
				@brief Gets the workspace identifier for the menu item.
				@param outIdentifier identifier in use for this menu item.
				@return kVCOMError_NoError if the operation was successful.

				@author Lyndsey Ferguson
			 */
			virtual VCOMError VCOM_CALLTYPE GetIdentifier(TXString& outIdentifier) = 0;
			
			/**
				@brief Sets the universal name for the tool.
				@param inName the universal name for this tool.
				@return kVCOMError_NoError if the operation was successful.

				@author Lyndsey Ferguson
			 */
			virtual VCOMError VCOM_CALLTYPE SetUniversalName(const TXString& inName) = 0;
			
			/**
				@brief Gets the universal name for the menu item.
				@param outName on output, if successful, the universal name for this menu item.
				@return kVCOMError_NoError if the operation was successful.

				@author Lyndsey Ferguson
			 */
			virtual VCOMError VCOM_CALLTYPE GetUniversalName(TXString& outName) = 0;
			
			/**
				@brief Sets the Resource Manager ID for this item.
				@param inResourceID the Resource Manager ID to save.
				@return kVCOMError_NoError if the operation was successful.

				@author Lyndsey Ferguson
			 */
			virtual VCOMError VCOM_CALLTYPE SetResourceManagerID(Sint16 inResourceID) = 0;

			/**
				@brief Gets the Resource Manager ID if there is one.
				@param outResourceManagerID is the resource manager file id.
				@return kVCOMError_NoError if the menu has a Resource Manager ID.

				@author Lyndsey Ferguson
			 */
			virtual VCOMError VCOM_CALLTYPE GetResourceManagerID(Sint16& outResourceManagerID) = 0;
							
			/**
				@brief Set the shortcut key information for this menu item.
				@param inShortcutKey the short cut key that will activate this menu item.
				@param inShortcutKeyModifiers the accompanying necessary modifier keys required to activate this menu item.
				@return kVCOMError_NoError if the operation was successful.

				@author Lyndsey Ferguson
			 */
			virtual VCOMError VCOM_CALLTYPE SetShortcutKeyInfo(char inShortcutKey, Uint32 inShortcutKeyModifiers) = 0;

			/**
				@brief Delete shortcut key for this menu item.
				@return kVCOMError_NoError if the operation was successful.
			 */
			virtual VCOMError VCOM_CALLTYPE DeleteShortcutKey() = 0;

			/**
				@brief Set the shortcut key information for this menu item. This will delete any submenu information as well as chunk status.
				@param outShortcutKey on output, if successful, the short cut key that will activate this menu item. May be NULL if the client is not interested in the data.
				@param outShortcutKeyModifiers on output, if successful, the accompanying necessary modifier keys required to activate this menu item. May be NULL if the client is not interested in the data.
				@return kVCOMError_NoError if the operation was successful.

				@author Lyndsey Ferguson
			 */
			virtual VCOMError VCOM_CALLTYPE GetShortcutKeyInfo(char& outShortcutKey, Uint32& outShortcutKeyModifiers) = 0; 
			
			/**
				@brief Sets the sub menu for this menu item. This will delete any short cut key information as well as menu chunk status.
				@param inSubMenu the menu that should be the submenu for this menu item.
				@return kVCOMError_NoError if the operation was successful.

				@author Lyndsey Ferguson
			 */
			virtual VCOMError VCOM_CALLTYPE SetSubMenu(IWorkspaceMenu* inSubMenu) = 0;
			/**
				@brief Gets the sub menu for this menu item.
				@param outSubMenu on output, if successful, a pointer to the submenu for this menu item.
				@return kVCOMError_NoError if the operation was successful.

				@author Lyndsey Ferguson
			 */
			virtual VCOMError VCOM_CALLTYPE GetSubMenu(IWorkspaceMenu** outSubMenu) = 0;
			/**
				@brief Gets the last error that was generated by the IWorkspaces interface.
				@param outError is the last error that occurred. Any successful call to a IWorkspaces function will reset this to EWorkspaces_NoError.
			 
				@author Lyndsey Ferguson
			 */
			virtual VCOMError VCOM_CALLTYPE GetLastError(EWorkspacesError& outError) = 0;
	
		};

				
		/**
			@class IWorkspaceMenuChunk

			This is a variation of the Menu Item that actually represents a group of menu items.

			A menu chunk can be:
			- a regular, static list of menu items.
			- a dynamic list of menu items ...?
			- a dynamic, variable list of menu items.
		*/
		class DYNAMIC_ATTRIBUTE IWorkspaceMenuChunk : public IVWUnknown
		{
		public:
			virtual ~IWorkspaceMenuChunk() {}
			/**
				@brief Gets the workspace identifier for the menu item.
				@param outIdentifier identifier in use for this menu item.
				@return kVCOMError_NoError if the operation was successful.

				@author Lyndsey Ferguson
			 */
			virtual VCOMError VCOM_CALLTYPE GetIdentifier(TXString& outIdentifier) = 0;
			
			/**
				@brief Sets the universal name for the menu chunk.
				@param inName the universal name for this menu chunk.
				@return kVCOMError_NoError if the operation was successful.

				@author Lyndsey Ferguson
			 */
			virtual VCOMError VCOM_CALLTYPE SetUniversalName(const TXString& inName) = 0;

			/**
				@brief Gets the universal name for the menu item.
				@param outName on output, if successful, the universal name for this menu item.
				@return kVCOMError_NoError if the operation was successful.

				@author Lyndsey Ferguson
			 */
			virtual VCOMError VCOM_CALLTYPE GetUniversalName(TXString& outName) = 0;
			
			/**
				@brief Sets the Resource Manager ID for this item.
				@param inResourceID the Resource Manager ID to save.
				@return kVCOMError_NoError if the operation was successful.
			 
				@author Lyndsey Ferguson
			*/
			virtual VCOMError VCOM_CALLTYPE SetResourceManagerID(Sint16 inResourceID) = 0;
			
			/**
				@brief Gets the Resource Manager ID if there is one.
			 	@param outResourceID is the resource manager file id.
				@return kVCOMError_NoError if the menu has a Resource Manager ID.

				@author Lyndsey Ferguson
			 */
			virtual VCOMError VCOM_CALLTYPE GetResourceManagerID(Sint16& outResourceID) = 0;
			
			/**
				@brief Gets the type of chunk that this menu chunk is.
				@param outType the menu chunk type of this menu chunk. 
				@return kVCOMError_NoError if the operation was successful.

				@author Lyndsey Ferguson
				@see EWorkspaceMenuChunkType.
			 */
			virtual VCOMError VCOM_CALLTYPE	GetChunkType(EWorkspaceMenuChunkType& outType) = 0;
			
			/**
				@brief Adds to the collection of menu items in this chunk.
				@param inMenuItemDisplayText the localized text used for this menu item.
				@param inHasShortcutKey if there is a short cut key and modifier for this item, this will be true. Defaults to false.
				@param inShortcutKey the short cut key that will activate this menu item.
				@param inShortcutKeyModifiers the accompanying necessary modifier keys required to activate this menu item.
				@return kVCOMError_NoError if the operation was successful.

				@author Lyndsey Ferguson
			*/
			virtual VCOMError VCOM_CALLTYPE AddMenuItem(const TXString& inMenuItemDisplayText, bool inHasShortcutKey = false, char inShortcutKey = (char) 0, Uint32 inShortcutKeyModifiers = 0UL) = 0;
			
			/**
				@brief Deletes menu items in this chunk.
				@return kVCOMError_NoError if the operation was successful.
				@author Payne Chang
			 */
			virtual VCOMError VCOM_CALLTYPE DeleteMenuItems() = 0;
			
			/**
				@brief Gets the count of menu itemss in the menu.
				@param outMenuItemCount on output, if successful, the number of menu items in the menu.
				@return kVCOMError_NoError if the operation was successful.

				@author Lyndsey Ferguson
			 */
			virtual VCOMError VCOM_CALLTYPE GetMenuItemCount(Sint16& outMenuItemCount) = 0;
			
			/**
				@brief Gets the menu item info at the given index.
				@param inMenuItemIndex the index of the menu item's info to retrieve.
				@param outMenuItemDisplayText on output, if successful, a pointer to a menu item pointer.
				@param outHasShortcutKey will be true if this menu item has shortcut key information to retrieve.
				@param outShortcutKey will contain the short cut key used for the menu item if outHasShortcutKey is true. Otherwise the value is undefined.
				@param outShortcutKeyModifiers will contain the short cut key modifiers used for the menu item if outHasShortcutKey is true. Otherwise the value is undefined.
				@return kVCOMError_NoError if the operation was successful.

				@author Lyndsey Ferguson
			 */
			virtual VCOMError VCOM_CALLTYPE GetMenuInfoAtIndex(Sint16 inMenuItemIndex, TXString& outMenuItemDisplayText, bool& outHasShortcutKey, char& outShortcutKey, Uint32& outShortcutKeyModifiers) = 0;
			
			/**
				@brief Sets shortcut key info at the given index.
				@param inMenuItemIndex the index of the menu item.
				@param inShortcutKey contains the shortcut key used for the menu item.
				@param inShortcutKeyModifier contains the shortcut key modifiers used for the menu item.
				@return kVCOMError_NoError if the operation was successful.
			 */
			virtual VCOMError VCOM_CALLTYPE SetShortcutKeyAtIndex(Sint16 inMenuItemIndex, const char& inShortcutKey, const Uint32& inShortcutKeyModifier) = 0;
			
			/**
				@brief Delete shortcut key at the given index.
				@param inMenuItemIndex the index of the menu item.
				@return kVCOMError_NoError if the operation was successful.
			 */
			virtual VCOMError VCOM_CALLTYPE DeleteShortcutKeyAtIndex(Sint16 inMenuItemIndex) = 0;
			
			 /**
				@brief Gets the last error that was generated by the IWorkspaces interface.
				@param outError is the last error that occurred. Any successful call to a IWorkspaces function will reset this to EWorkspaces_NoError.

				@author Lyndsey Ferguson
			 */
			virtual VCOMError VCOM_CALLTYPE GetLastError(EWorkspacesError& outError) = 0;
		};
		
		/**
			@class IWorkspaceMenu
		*/

		class DYNAMIC_ATTRIBUTE IWorkspaceMenu : public IVWUnknown
		{
		public:
			virtual ~IWorkspaceMenu() {}
			
			/**
				@brief Gets the identifier for the menu.
				@param outIdentifier is the identifier for this menu.
				@return kVCOMError_NoError if the operation was successful.

				@author Lyndsey Ferguson
			 */
			virtual VCOMError VCOM_CALLTYPE GetIdentifier(TXString& outIdentifier) = 0;
			
			/**
				@brief Sets the Resource Manager ID for this item.
				@param inResourceID the Resource Manager ID to save.
				@return kVCOMError_NoError if the operation was successful.

				@author Lyndsey Ferguson
			 */
			virtual VCOMError VCOM_CALLTYPE SetResourceManagerID(Sint16 inResourceID) = 0;

			/**
				@brief Gets the Resource Manager ID if there is one.
				@param outResourceManagerID is the resource manager file id.
				@return kVCOMError_NoError if the menu has a Resource Manager ID.
			 
				@author Lyndsey Ferguson
			*/
			virtual VCOMError VCOM_CALLTYPE GetResourceManagerID(Sint16& outResourceManagerID) = 0;	
			
			/**
				@brief Sets the display name for the menu.
				@param inDisplayName is the display name to be used in the GUI.
				@return kVCOMError_NoError if the operation was successful.
			 
				@author Lyndsey Ferguson
			*/
			virtual VCOMError VCOM_CALLTYPE SetDisplayName(const TXString& inDisplayName) = 0;
			
			/**
				@brief Gets the display name for the menu.
				@param outDisplayName is the display name used in the GUI.
				@return kVCOMError_NoError if the operation was successful.

				@author Lyndsey Ferguson
			 */
			virtual VCOMError VCOM_CALLTYPE GetDisplayName(TXString& outDisplayName) = 0;
			
			/**
				@brief Adds a menu item to this menu.
				@param inMenuItem a reference to a menu item or a menu chunk.
				@return kVCOMError_NoError if the operation was successful.

				@author Lyndsey Ferguson
			 */
			virtual VCOMError VCOM_CALLTYPE AddMenuItem(IVWUnknown* inMenuItem, EWorkspaceMenuChunkType inChunkType = eWorkspaceMenuChunkType_None) = 0;

			/**
				@brief Adds a menu item to this menu before index.
				@param inMenuItem a reference to a menu item or a menu chunk.
				@param inChunkType menu chunk type.
				@param inIndex menu item is added before this index. index is ignored if less than 0.
				@return kVCOMError_NoError if the operation was successful.
			 
				@author Payne Chang
			 */
			virtual VCOMError VCOM_CALLTYPE AddMenuItemBeforeIndex(IVWUnknown* inMenuItem, EWorkspaceMenuChunkType inChunkType = eWorkspaceMenuChunkType_None, const Sint16 inIndex = -1) = 0;
			
			/**
				@brief Gets the count of menu itemss in the menu.
				@param outMenuItemCount on output, if successful, the number of menu items in the menu.
				@return kVCOMError_NoError if the operation was successful.
			 
				@author Lyndsey Ferguson
			 */
			virtual VCOMError VCOM_CALLTYPE GetMenuItemCount(Sint16& outMenuItemCount) = 0;
			
			/**
				@brief Gets menu item at the given index.
				@param inMenuItemIndex the index of the menu item to retrieve.
				@param outMenuItem on output, if successful, a pointer to a menu item pointer or menu chunk pointer.
			 	@param outChunkType on output, the chunck type
				@return kVCOMError_NoError if the operation was successful.

				@author Lyndsey Ferguson
			*/
			 virtual VCOMError VCOM_CALLTYPE GetMenuItemAtIndex(Sint16 inMenuItemIndex, IVWUnknown** outMenuItem, EWorkspaceMenuChunkType& outChunkType) = 0;
			 
			 /**
				@brief Gets the last error that was generated by the IWorkspaces interface.
				@param outError is the last error that occurred. Any successful call to a IWorkspaces function will reset this to EWorkspaces_NoError.
			 
				@author Lyndsey Ferguson
			 */
			virtual VCOMError VCOM_CALLTYPE GetLastError(EWorkspacesError& outError) = 0;
	
			/**
				@brief Removes a menu item at the given index.
				@param inMenuItemIndex the index of the menu item to retrieve.
				@return kVCOMError_NoError if the operation was successful.

				@author Vladislav Stanev
			*/
			 virtual VCOMError VCOM_CALLTYPE RemoveMenuItemAtIndex(Sint16 inMenuItemIndex) = 0;
		};

		/**
		@class IWorkspaceBar
		*/


		class DYNAMIC_ATTRIBUTE IWorkspaceMenuBar : public IVWUnknown
		{
		public:
			virtual ~IWorkspaceMenuBar() {}
			
			/**
				@brief Adds a menu to this menu bar.
				@param inMenu a reference to a menu.
				@param beforeIndex before existing index. -1 would be at the end of the list
				@return kVCOMError_NoError if the operation was successful.
			 
				@author Lyndsey Ferguson
			*/
			virtual VCOMError VCOM_CALLTYPE AddMenu(IWorkspaceMenu* inMenu, Sint16 beforeIndex=-1) = 0;
			/**
				@brief Gets the count of menus in the menu bar.
				@param outMenuCount on output, if successful, the number of menus in the menu bar.
				@return kVCOMError_NoError if the operation was successful.
			 
				@author Lyndsey Ferguson
			 */
			virtual VCOMError VCOM_CALLTYPE GetMenuCount(Sint16& outMenuCount) = 0;
			/**
				@brief Gets menu at the given index.
				@param inIndex the index of the menu to retrieve.
				@param outMenu on output, if successful, a pointer to a menu pointer.
				@return kVCOMError_NoError if the operation was successful.

				@author Lyndsey Ferguson
			 */
			virtual VCOMError VCOM_CALLTYPE GetMenuAtIndex(Sint16 inIndex, IWorkspaceMenu** outMenu) = 0;
			/**
				@brief Remove the menu at the given index.
				@param inIndex the index of the menu to retrieve.
				@return kVCOMError_NoError if the operation was successful.

				@author Vladislav Stanev
			 */
			virtual VCOMError VCOM_CALLTYPE RemoveMenuAtIndex(Sint16 inIndex) = 0;
			/**
				@brief Gets the last error that was generated by the IWorkspaces interface.
				@param outError is the last error that occurred. Any successful call to a IWorkspaces function will reset this to EWorkspaces_NoError.
			 
				@author Lyndsey Ferguson
			 */
			virtual VCOMError VCOM_CALLTYPE GetLastError(EWorkspacesError& outError) = 0;

		};

		/**
			@enum ESpecialKeys
		 
			@brief Defines the shortcut key ids associated with the miscellanious workspace keys.
		*/
		enum ESpecialKeys {
			eSpecialKeys_GroupMode1,
			eSpecialKeys_GroupMode2,
			eSpecialKeys_GroupMode3,
			eSpecialKeys_GroupMode4,
			eSpecialKeys_GroupMode5,
			eSpecialKeys_GroupMode6,
			eSpecialKeys_ScreenHintsToggle,
			eSpecialKeys_SetDatum,
			eSpecialKeys_SetSmartPointEdgeVectorSnap,
			eSpecialKeys_SnapLoupe,
			eSpecialKeys_RelativeAnglesToggle,
			eSpecialKeys_SuspendSnapping,
			eSpecialKeys_AutomaticWorkingPlaneToggle,
			eSpecialKeys_XRayEdgeSelection,
			eSpecialKeys_CoincidentSelection,
			eSpecialKeys_ClosePoly,
			eSpecialKeys_ToggleMultipleViewPanes,
			eSpecialKeys_TriggerOnDemandMenu,
			eSpecialKeys_LaunchGlobalSearch,
			eSpecialKeys_RunLastCommand,
			eSpecialKeys_NumItems
		};

		/**
			@struct TSpecialKeys
			@brief This legacy structure served as a wrapper around an array of keys for the various mode bar short cut keys.
		 
			Do not use this for version 2009 or later. Instead, use GetSpecialKey.
		*/
		struct NNA_DEPRECATED TSpecialKeys {
			char key[9];
		};
		
		/**
			@enum EConstraintsKeys
		 
			@brief Defines the shortcut key ids associated with the constraints workspace keys.
		 */
		enum EConstraintsKeys {
			eConstraintsKeys_SnapToGrid,
			eConstraintsKeys_SnapToObject,
			eConstraintsKeys_SnapToAngle,
			eConstraintsKeys_SnapToIntersection,
			eConstraintsKeys_SnapToPoints,
			eConstraintsKeys_SnapToDistance,
			eConstraintsKeys_SnapToEdge,
			eConstraintsKeys_SnapToTangent,
			eConstraintsKeys_SnapToWorkingPlane,
			eConstraintsKeys_NumItems
		};
		
		/**
			@struct TConstraintsKeys
			@brief This structure serves as a wrapper around an array of keys for the various constraints short cut keys.
		*/
		struct NNA_DEPRECATED TConstraintsKeys {
			char key[9];
		};
		
		const Sint32 kWorkspaceFileVersion_Current = 3L;

		/**
		 	@brief IID_WorkspaceFile
		 	{E0574BF4-85C8-4782-8F4B-58F7E8A0B2C0}
		*/
		static const VWIID IID_WorkspaceFile = { 0xE0574BF4, 0x85C8, 0x4782, { 0x8F, 0x4B, 0x58, 0xF7, 0xE8, 0xA0, 0xB2, 0xC0 } };

		class DYNAMIC_ATTRIBUTE IWorkspaceFile : public IVWUnknown
		{
		public:
			virtual ~IWorkspaceFile() {}
			
			/**
				@brief Creates a new Workspace read to be saved.
				@param inWorkspaceFileToCreate the new workspace name.
				@return kVCOMError_NoError if the operation was successful.
			 
				@author Lyndsey Ferguson
				@date 11/01/2006
			 */
			virtual VCOMError VCOM_CALLTYPE Create(const TXString& inWorkspaceFileToCreate) = 0;
			
			/**
				@func Open
				@param inWorkspaceFileToOpen is the name of the workspace file that VectorWorks open.
				@param inPermissions is the read/write permission bits that should be used to open the workspace file.
				@return kVCOMError_NoError if the operation was successful.

				@author Lyndsey Ferguson
				@date 10/02/2006
			*/
			virtual VCOMError VCOM_CALLTYPE Open(const TXString& inWorkspaceFileToOpen, Sint32 inPermissions) = 0;
			
			/**
				Determines if the workspace file has been opened or not.
				@param openStatus on output, if successful, true if the workspace has been opened. False otherwise.
				@return kVCOMError_NoError if the operation was successful.
			 
				@author Lyndsey Ferguson
				@date 12/08/2006
			*/
			virtual VCOMError VCOM_CALLTYPE IsOpen(bool& openStatus) = 0;

			/**
				Determines if the workspace file is writable or not.
				@param bWritable on output, if successful, true if the workspace is writable. False otherwise.
				@return kVCOMError_NoError if the operation was successful.
			*/
			virtual VCOMError VCOM_CALLTYPE IsWritable(bool& bWritable) = 0;
			
			/**
				@func Close
				@return kVCOMError_NoError if the operation was successful.

				@author Lyndsey Ferguson
				@date 10/02/2006
			 */
			virtual VCOMError VCOM_CALLTYPE Close() = 0;
			
			/**
				@brief Saves the workspace.
				@param inSavedFileName which defaults to an empty string if the same name used to open it is to be used.
				@return kVCOMError_NoError if the operation was successful.
				@author Lyndsey Ferguson
				@date 10/02/2006
			*/
			virtual VCOMError VCOM_CALLTYPE Save(const TXString& inSavedFileName = "") = 0;
			
			/**
				@func SetVersion
				@param inWorkspaceVersionCurrent is the number of the workspace file revision.
				@return kVCOMError_NoError if the operation was successful.

				@author Lyndsey Ferguson
				@date 10/02/2006
			 */
			virtual VCOMError VCOM_CALLTYPE SetVersion(Sint32 inWorkspaceVersionCurrent) = 0;
			/**
				@func GetVersion

				@param outWorkspaceFileVersion on output, if successful, the workspace file version.
				@return kVCOMError_NoError if the operation was successful.

				@author Lyndsey Ferguson
				@date 10/02/2006
			 */
			virtual VCOMError VCOM_CALLTYPE GetVersion(Sint32& outWorkspaceFileVersion) = 0;
			/**
				@func GetMenuBar
				@brief Gets the menu bar for the current workspace.
				@param outMenuBar on output, if successful, a pointer to the menu bar that exists in this workspace.
				@return kVCOMError_NoError if the operation was successful.
			 
				@author Lyndsey Ferguson
			*/
			virtual VCOMError VCOM_CALLTYPE GetMenuBar(IWorkspaceMenuBar** outMenuBar) = 0;

			/**
				@brief Sets the various shortcut keys for the workspace.
			 
				@param inSpecialKeys the mode bar keys (such as enter/exit group, mode bar button iteration, etc)
				@param inConstraintsKeys the shortcut keys to toggle the various constraints on or off.
				@return kVCOMError_NoError if the operation was successful.

				@author Lyndsey Ferguson
				@date 10/02/2006
			 */
			virtual VCOMError VCOM_CALLTYPE NNA_DEPRECATED SetShortcutKeys(TSpecialKeys inSpecialKeys, TConstraintsKeys inConstraintsKeys) = 0;
			
			/**
				@brief Gets the various shortcut keys for the workspace. Any parameter may be NULL if the client is not interested in a specific piece of data. The display string for the keycodes returned in outSpecialKeys and outConstraintsKeys can be converted into an equivalent string form with the GetKeyDisplayString function.
			 
				@param outSpecialKeys on output, if successful, the mode bar keys (such as enter/exit group, mode bar button iteration, etc)
				@param outConstraintsKeys on output, if successful, the shortcut keys to toggle the various constraints on or off.
				@return kVCOMError_NoError if the operation was successful.

				@author Lyndsey Ferguson
				@date 10/02/2006
			 */
			virtual VCOMError VCOM_CALLTYPE NNA_DEPRECATED GetShortcutKeys(TSpecialKeys& outSpecialKeys, TConstraintsKeys& outConstraintsKeys) = 0;
						
			/**
				@brief Sets the OS data specific to the Windows palette (initially the BCGBPro data).
				@param inData the XML data used by the Windows OS.
				@return kVCOMError_NoError if the operation was successful.

				@author Lyndsey Ferguson
				@date 10/02/2006
			 */
			virtual VCOMError VCOM_CALLTYPE SetWindowsOSData(const TXString& inData) = 0;
			
			/**
				@brief Gets the OS data specific to the Windows palette (BCBPro data)
				@param outData a string containing the XML data used by the Windows OS.
				@return kVCOMError_NoError if the operation was successful.

				@author Lyndsey Ferguson
				@date 10/02/2006
			 */
			virtual VCOMError VCOM_CALLTYPE GetWindowsOSData(TXString& outData) = 0;

			/**
				@brief Sets a set of dimensions that represent either the Application Frame (for Windows) or the size of the Main Display (for Macintosh).
				@param inTop the top of the frame in relation to the Monitor (for Mac, this will always be 0).
				@param inLeft the left of the frame in relation to the Monitor (for Mac, this will always be 0).
				@param inWidth the width of the Application Frame or Monitor.
				@param inHeight the height of the Application Frame or Monitor.
			 
				@author Lyndsey Ferguson
				@date 12/08/2006
			*/
			virtual VCOMError VCOM_CALLTYPE SetApplicationOrMonitorRect(Sint16 inTop, Sint16 inLeft, Sint16 inWidth, Sint16 inHeight) = 0;
			
			/**
				@brief Gets a set of dimensions that represent either the Application Frame (for Windows) or the size of the Main Display (for Macintosh).
				@param outTop the top of the frame in relation to the Monitor (for Mac, this will always be 0).
				@param outLeft the left of the frame in relation to the Monitor (for Mac, this will always be 0).
				@param outWidth the width of the Application Frame or Monitor.
				@param outHeight the height of the Application Frame or Monitor.

				@author Lyndsey Ferguson
				@date 12/08/2006
			 */
			virtual VCOMError VCOM_CALLTYPE GetApplicationOrMonitorRect(Sint16& outTop, Sint16& outLeft, Sint16& outWidth, Sint16& outHeight) = 0;
			
			/**
				@brief Creates a new menu item in the workspace that can be added to a menu if it does not yet exist.
				@param inIdentifier the identifier to be used to represent this menu item. Use the IWorkspaces interface to create this identifier.
				@param outMenuItem will be a newly created IWorkspaceMenuItem* on output.
				@return kVCOMError_NoError if the operation was successful.
								
				@author Lyndsey Ferguson
				@date 10/09/2006
				@see IWorkspaces
			*/
			virtual VCOMError VCOM_CALLTYPE CreateMenuItem(const TXString& inIdentifier, IWorkspaceMenuItem** outMenuItem) = 0;
			
			/**
				@brief Creates a new menu chunk in the workspace that can be added to a menu if it does not yet exist.
				@param inIdentifier the identifier to be used to represent this menu chunk. Use the IWorkspaces interface to create this identifier.
				@param inChunkType is the type of chunk menu item to create.	
				@param outMenuChunk will be a newly created IWorkspaceMenuChunk* on output.
				@return kVCOMError_NoError if the operation was successful.

				@author Lyndsey Ferguson
				@date 10/09/2006
				@see IWorkspaces
			 */
			virtual VCOMError VCOM_CALLTYPE CreateMenuChunk(const TXString& inIdentifier, EWorkspaceMenuChunkType inChunkType, IWorkspaceMenuChunk** outMenuChunk) = 0;

			/**
				@brief Gets the menu, or if it does not exist, creates a new menu in the Workspace that can be referrenced either by the Menu Bar or a Menu Item or as a Menu Chunk.
				@param inIdentifier the identifier to be used to represent this menu. Use the IWorkspaces interface to create this identifier.
				@param outMenu on ouput, if successful a workspace menu VCOM interface.
				@return kVCOMError_NoError if the operation was successful.

				@author Lyndsey Ferguson
				@date 10/09/2006
				@see IWorkspaces
			*/
			virtual VCOMError VCOM_CALLTYPE CreateMenu(const TXString& inIdentifier, IWorkspaceMenu** outMenu) = 0;

			/**
				@brief Creates a tool in the workspace if it does not exist.
				@param inIdentifier the identifier to be used to represent this tool. Use the IWorkspaces interface to create this identifier.
				@param outTool will be the newly created tool in the workspace.
				@return kVCOMError_NoError if the operation was successful.
			 
				@author Lyndsey Ferguson
				@date 10/18/2006
				@see IWorkspaces
			*/
			virtual VCOMError VCOM_CALLTYPE CreateTool(const TXString& inIdentifier, IWorkspaceTool** outTool) = 0;
			
			/**
				@brief Creates a tool set in the workspace if it does not exist.
				@param inIdentifier the identifier to be used to represent this tool set. Use the IWorkspaces interface to create this identifier.
				@param outToolSet the newly created tool palette.
				@return kVCOMError_NoError if the operation was successful.
			 
				@author Lyndsey Ferguson
				@date 10/18/2006
				@see IWorkspaces
			*/
			virtual VCOMError VCOM_CALLTYPE CreateToolSet(const TXString& inIdentifier, IWorkspaceToolSet** outToolSet) = 0;
			
			/**
				@brief creates a tool palette in the workspace if it does not exist.
				@param inIdentifier the identifier to be used to represent this tool palette. Use the IWorkspaces interface to create this identifier.
				@param outToolPalette the created tool palette.
				@return kVCOMError_NoError if the operation was successful.
			 
				@author Lyndsey Ferguson
				@date 10/09/2006
				@see IWorkspaces
			 */
			virtual VCOMError VCOM_CALLTYPE CreateToolPalette(const TXString& inIdentifier, IWorkspaceToolPalette** outToolPalette) = 0;
			
			/**
				@brief Gets the tool that corresponds to the given identifier.
				@param inIdentifier the identifier that represents the desired tool.
				@param outTool on output, if successful, a pointer to the tool.
				@return kVCOMError_NoError if the operation was successful.
			 
				@author Lyndsey Ferguson
				@date 11/01/2006
				@see IWorkspaces
			 */
			virtual VCOMError VCOM_CALLTYPE GetToolUsingIdentifier(const TXString& inIdentifier, IWorkspaceTool** outTool) = 0;
						
			/**
				@brief Gets the tool set that corresponds to the given identifier.
				@parma inIdentifier the identifier that represents the desired tool set.
				@param outToolSet on output, if successful, a pointer to the tool set.
				@return kVCOMError_NoError if the operation was successful.

				@author Lyndsey Ferguson
				@data 11/08/2006
			*/
			virtual VCOMError VCOM_CALLTYPE GetToolSetUsingIdentifier(const TXString& inIdentifier, IWorkspaceToolSet** outToolSet) = 0;

			/**
				@brief Gets the tool palette that corresponds to the given identifier.
				@parma inIdentifier the identifier that represents the desired tool palette.
				@param outToolPalette on output, if successful, a pointer to the tool palette.
				@return kVCOMError_NoError if the operation was successful.

				@author Lyndsey Ferguson
				@data 11/08/2006
			 */
			virtual VCOMError VCOM_CALLTYPE GetToolPaletteUsingIdentifier(const TXString& inIdentifier, IWorkspaceToolPalette** outToolPalette) = 0;
			
			/**
				@brief Gets the menu item that corresponds to the given identifier.
				@param inIdentifier the identifier that represents the desired menu item.
				@param outMenuItem on output, if successful, a pointer to the menu item.
				@return kVCOMError_NoError if the operation was successful.

				@author Lyndsey Ferguson
				@date 11/01/2006
			 */
			virtual VCOMError VCOM_CALLTYPE GetMenuItemUsingIdentifier(const TXString& inIdentifier, IWorkspaceMenuItem** outMenuItem) = 0;
			
			/**
				@brief Gets the menu chunk that corresponds to the given identifier.
				@param inIdentifier the identifier that represents the desired menu chunk.
				@param outMenuChunk on output, if successful, a pointer to the menu chunk.
			 
				@author Lyndsey Ferguson
				@date 11/01/2006
			*/
			virtual VCOMError VCOM_CALLTYPE GetMenuChunkUsingIdentifier(const TXString& inIdentifier, IWorkspaceMenuChunk** outMenuChunk) = 0;
			
			/**
				@brief Gets the menu that corresponds to the given identifier.
				@param inIdentifier the identifier that represents the desired menu.
				@param outMenu on output, if successful, a pointer to the menu.
				@return kVCOMError_NoError if the operation was successful.
			 
				@author Lyndsey Ferguson
				@date 11/03/2006
			*/
			virtual VCOMError VCOM_CALLTYPE GetMenuUsingIdentifier(const TXString& inIdentifier, IWorkspaceMenu** outMenu) = 0;
				
			/**
				@brief Removes the menu that corresponds to the given identifier.
				@param inIdentifier the identifier that represents the desired menu.
				@return kVCOMError_NoError if the operation was successful.
			 
				@author Vladislav Stanev
				@date 4/08/2020
			*/
			virtual VCOMError VCOM_CALLTYPE RemoveMenuUsingIdentifier(const TXString& inIdentifier) = 0;
		
			/**
				@brief Removes the menu item or menu chunk item that corresponds to the given identifier.
				@param inIdentifier the identifier that represents the desired menu.
				@return kVCOMError_NoError if the operation was successful.
			 
				@author Vladislav Stanev
				@date 4/08/2020
			*/
			virtual VCOMError VCOM_CALLTYPE RemoveMenuItemUsingIdentifier(const TXString& inIdentifier) = 0;

			/**
				@brief Gets the count of tool palettes in the workspace.
				@param outCount on output, if successful, the number of tool palettes in the workspace.
				@return kVCOMError_NoError if the operation was successful.
			 
				@author Lyndsey Ferguson
			*/
			virtual VCOMError VCOM_CALLTYPE GetToolPaletteCount(Sint16& outCount) = 0;
			
			/**
				@brief Gets the tool palette at the given index.
				@param inIndex the index into the list of palettes that we are interested in.
				@param outToolPalette on output, if successful, the tool palette at the given index.
				@return kVCOMError_NoError if the operation was successful.
			 
				@author Lyndsey Ferguson
			 */
			virtual VCOMError VCOM_CALLTYPE GetToolPaletteAtIndex(Sint16 inIndex, IWorkspaceToolPalette** outToolPalette) = 0;

			/**
				@brief Deletes the given tool palette from the workspace and sets the IWorkspaceToolPalette NULL.
				@param ioToolPalette on output, if successful, a pointer to a NULL pointer.
				@return kVCOMError_NoError if the operation was successful.
			 
				@author Lyndsey Ferguson
			 */
			virtual VCOMError VCOM_CALLTYPE DeleteToolPalette(IWorkspaceToolPalette** ioToolPalette) = 0;

			/**
				@brief Removes a toolset
				@param ioToolSet the toolset to be removed
				@return kVCOMError_NoError if the operation was successful.

				@author Vladislav Stanev
			 */
			virtual VCOMError VCOM_CALLTYPE DeleteToolSet(IWorkspaceToolSet** ioToolSet) = 0;

			/**
				@brief Removes a tool
				@param ioTool the tool to be removed
				@return kVCOMError_NoError if the operation was successful.

				@author Vladislav Stanev
			 */
			virtual VCOMError VCOM_CALLTYPE DeleteTool(IWorkspaceTool** ioTool) = 0;

			/**
				@brief Gets the attributes palette.
				@param outAttributesPalette on output, if successful, the atttributes palette.
				@return kVCOMError_NoError if the operation was successful.

				@author Lyndsey Ferguson
			*/
			virtual VCOMError VCOM_CALLTYPE GetAttributesPalette(IWorkspaceAttributesPalette** outAttributesPalette) = 0;
			
			/**
				@brief Gets the object info palette.
				@param outObjectInfoPalette on output, if successful, the object info palette.
				@return kVCOMError_NoError if the operation was successful.

				@author Lyndsey Ferguson
			 */
			virtual VCOMError VCOM_CALLTYPE GetObjectInfoPalette(IWorkspaceObjectInfoPalette** outObjectInfoPalette) = 0;

			/**
			 @brief Get palette node for the new detachable tab palettes.
			 @param outPalette The palette node which has the specified palette ID.
			 @param	paletteID The palette ID of the palette node.
			 @return VCOM error code.
			 */
			virtual VCOMError VCOM_CALLTYPE GetNewSystemPalette(IWorkspaceObjectInfoPalette** outPalette, Sint16 paletteID) = 0;
			virtual VCOMError VCOM_CALLTYPE GetNewSystemPalette(IWorkspaceNavigationPalette** outPalette, Sint16 paletteID) = 0;
			virtual VCOMError VCOM_CALLTYPE GetNewSystemPalette(IWorkspaceVisualizationPalette** outPalette, Sint16 paletteID) = 0;

			/**
				@brief Gets the message palette.
				@param outMessagePalette on output, if successful, the message palette.
				@return kVCOMError_NoError if the operation was successful.

				@author Lyndsey Ferguson
			 */
			virtual VCOMError VCOM_CALLTYPE GetMessagePalette(IWorkspaceMessagePalette** outMessagePalette) = 0;

			/**
				@brief Gets the navigation palette.
				@param outNavigationPalette on output, if successful, the navigation palette.
				@return kVCOMError_NoError if the operation was successful.

				@author Lyndsey Ferguson
			 */
			virtual VCOMError VCOM_CALLTYPE GetNavigationPalette(IWorkspaceNavigationPalette** outNavigationPalette) = 0;

			/**
				@brief Gets the visualization palette.
				@param outVisualizationPalette on output, if successful, the visualization palette.
				@return kVCOMError_NoError if the operation was successful.
			 */
			virtual VCOMError VCOM_CALLTYPE GetVisualizationPalette(IWorkspaceVisualizationPalette** outVisualizationPalette) = 0;
						
			
			virtual VCOMError VCOM_CALLTYPE GetResourceManagerPalette(IWorkspaceResourceManagerPalette** /*outResourceManager*/) = 0;	
			/**		
				@brief Gets the working plane palette.
				@param outWorkingPlanePalette on output, if successful, the working plane palette.
				@return kVCOMError_NoError if the operation was successful.
				 
				@author Lyndsey Ferguson
			*/
			virtual VCOMError VCOM_CALLTYPE GetWorkingPlanePalette(IWorkspaceWorkingPlanePalette** outWorkingPlanePalette) = 0;


			/**
				@brief Gets the Document Contextual Menu.
				@param outDocumentContextualMenu on output, if successful, the document contextual palette.
				@return kVCOMError_NoError if the operation was successful.
			 
				@author Lyndsey Ferguson
			*/
			virtual VCOMError VCOM_CALLTYPE GetDocumentContextualMenu(IWorkspaceMenu** outDocumentContextualMenu) = 0;
			
			/**
				@brief Gets the Object Contextual Menu.
				@param outObjectContextualMenu on output, if successful, the object contextual palette.
				@return kVCOMError_NoError if the operation was successful.

				@author Lyndsey Ferguson
			 */
			virtual VCOMError VCOM_CALLTYPE GetObjectContextualMenu(IWorkspaceMenu** outObjectContextualMenu) = 0;
			
			/**
				@brief Copies all palette state information from the given Workspace file.
				@param inSourceWorkspaceFile the workspace file to read the palette state info from.
				@return kVCOMError_NoError if the operation was successful.
			 
				@author Lyndsey Ferguson
			*/
			virtual VCOMError VCOM_CALLTYPE CopyPaletteStateInfoFromWorkspace(IWorkspaceFile* inSourceWorkspaceFile) = 0;
			
			/**
				@brief Gets the last error that was generated by the IWorkspaces interface.
				@param outError is the last error that occurred. Any successful call to a IWorkspaces function will reset this to EWorkspaces_NoError.
			 
				@author Lyndsey Ferguson
			 */
			virtual VCOMError VCOM_CALLTYPE GetLastError(EWorkspacesError& outError) = 0;
			
			// =========================================================================================================
			// New Functions as of VectorWorks 2009
			// =========================================================================================================
			
			/**
				@brief Gets the keycode for the given "special" key.
			
				@author Lyndsey Ferguson
			 */
			
			virtual VCOMError VCOM_CALLTYPE GetSpecialKey(ESpecialKeys specialKeyID, char& outKeyCode) = 0;
			
			/**
				@brief Sets the keycode for the given "special" key.
			 
				@author Lyndsey Ferguson
			 */
			virtual VCOMError VCOM_CALLTYPE SetSpecialKey(ESpecialKeys specialKeyID, char inKeyCode) = 0;

			/**
				@brief Gets the keycode for the given constraint key.
			 
				@author Lyndsey Ferguson
			 */
			
			virtual VCOMError VCOM_CALLTYPE GetConstraintKey(EConstraintsKeys constraintKeyID, char& outKeyCode) = 0;
			
			/**
				@brief Gets the keycode for the given constraint key.
			 
				@author Lyndsey Ferguson
			 */
			virtual VCOMError VCOM_CALLTYPE SetConstraintKey(EConstraintsKeys constraintKeyID, char inKeyCode) = 0;
			
			/**
				@brief Converts the keycodes obtained by the GetShortcutKeys function
			 
				@param inKeyCode is a keycode from TSpecialKeys or TConstraintsKeys
				@param outDisplayString on output, if successful, is the string equivalent of the keycode.
				@return kVCOMError_NoError if the operation was successful.

				@author Eric Gorr
				@date 21/05/2008
			 */
			virtual VCOMError VCOM_CALLTYPE GetKeyDisplayString( char inKeyCode, TXString &outDisplayString ) = 0;
	
			// =========================================================================================================
			// New Function as of VectorWorks 2012
			// =========================================================================================================

			/**
				@brief Converts a Mac keycode to a Win keycode

				@param macKeyCode is a Mac keycode
				@param outWinKeyCode on output, if successful, is the Win keycode that corresponds to the Mac keycode
				@param outExtended on output, if successful, indicates if the key is an extended key
				@return kVCOMError_NoError if the operation was successful.

				@author Unknown (added to interface by Bruce Ferguson)
			*/
			virtual VCOMError VCOM_CALLTYPE MacKeyToWinKey( char macKeyCode, char& outWinKeyCode, bool& outExtended ) = 0;
            
            virtual VCOMError VCOM_CALLTYPE GetVWWindowDockingNode(VCOMPtr<VectorWorks::Filing::IXMLFileNode>& n) = 0;
            
            virtual VCOMError VCOM_CALLTYPE GetFloatingWindowsDockingNode(VCOMPtr<VectorWorks::Filing::IXMLFileNode>& n) = 0;

		};
		
		/**
			@enum EWorkspaceItemType
		 
			The different graphical user interface items that appear in the workspace and can be managed.
		*/
		enum EWorkspaceItemType {
			eWorkspaceItemType_MenuItem,
			eWorkspaceItemType_MenuChunk,
			eWorkspaceItemType_Menu,
			eWorkspaceItemType_Tool,
			eWorkspaceItemType_ToolSet,
			eWorkspaceItemType_ToolPalette
		};

		/**
			@brief IWorkspaces
			{BD71F2D6-8C66-4D21-A613-3A92D7529945}
		*/
		static const VWIID IID_Workspaces = { 0xBD71F2D6, 0x8C66, 0x4D21, { 0xA6, 0x13, 0x3A, 0x92, 0xD7, 0x52, 0x99, 0x45 } };
		
		class DYNAMIC_ATTRIBUTE IWorkspaces : public IVWSingletonUnknown
		{
		public:
			virtual ~IWorkspaces() {}
			
			/**
				Using the given Resource Manager ID, creates the workspace identifier.
				@param inItemType is either a Menu Item, Menu Chunk, or Tool. No other workspace item type requires such identifiers.
				@param inResourceID is the internal resource id (a negative number).
				@param outIdentifier on output, if successful, the workspace identifier.
				@return kVCOMError_NoError if successful.
			 
				@author Lyndsey Ferguson
			*/
			virtual VCOMError VCOM_CALLTYPE CreateIdentifierUsingResourceID(EWorkspaceItemType inItemType, Sint16 inResourceID, TXString& outIdentifier) = 0;
			
			/**
				Using the given Universal Name, creates the workspace identifier.
				@param inItemType is either a Menu Item or Tool. No other workspace item type requires such identifiers.
				@param inUniversalName is the universal name for the workspace item.
				@param outIdentifier on output, if successful, the workspace identifier.
				@return kVCOMError_NoError if successful.
			*/
			virtual VCOMError VCOM_CALLTYPE CreateIdentifierUsingUniversalName(EWorkspaceItemType inItemType, const TXString& inUniversalName, TXString& outIdentifier) = 0;
			
			/**
				Using the given workspace item type, creates a unique workspace identifier.
				@param inItemType is either a Menu, Menu Chunk, Tool Palette, or Tool Set.  No other workspace item type requires such identifiers.
				@param outIdentifier on output, if successful, the workspace identifier.
				@return kVCOMError_NoError if successful.
			 
				@author Lyndsey Ferguson
			*/
			virtual VCOMError VCOM_CALLTYPE CreateIdentifier(EWorkspaceItemType inItemType, TXString& outIdentifier) = 0;
			
			/**
				Converts a string to one that will be safe to store in the XML file. Bad strings in the XML file can cause exceptions to be thrown.
				@param ioString on input, the normalized string, on output, the xml-safe string.
				@return kVCOMError_NoError if there was no error.
			*/
			virtual VCOMError VCOM_CALLTYPE ConvertStringToSafeXMLFormat(TXString& ioString) = 0;
			
			/**
				Converts a xml-safe string to its original, normalized state.
				@param ioString on input, the xml-safe string, on output, the normalized string.
				@return kVCOMError_NoError if there was no error.
			*/
			virtual VCOMError VCOM_CALLTYPE ConvertSafeXMLFormatToString(TXString& ioString) = 0;
			
			/**
				Gets the image data (encoded as a PNG) and size given the Workspace Identifier.
			 */
			virtual VCOMError VCOM_CALLTYPE GetIcon(const TXString& inIdentifier, VectorWorks::UI::IIcon* outIcon) = 0;

			/**
				Gets the item's display/localized name given the Workspace Identifier.
			 */
			virtual VCOMError VCOM_CALLTYPE GetDisplayName(const TXString& inIdentifier, TXString& outDisplayName) = 0;

			/**
				Copies associated palette settings that are saved in the user's preferences.
			*/
			virtual VCOMError VCOM_CALLTYPE CopyPaletteSettingsFromUserPreferencesToWorkspace(const TXString& fromWorkspaceName, IWorkspaceFile* workspaceFile) = 0;


			// This enum describes where a special menu item should be placed
			enum ESpecialMenuItemLocation {
				eSpecialMenuItemLocation_Help,			// the help menu (all platforms)
				eSpecialMenuItemLocation_Application,	// the application menu (mac only)
			};

			// These exist so that integer indices into an array are not confused
			static const Uint32 kSpecialMenuOS_MacOS				= 0;
			static const Uint32 kSpecialMenuOS_Windows				= 1;
			static const Uint32 kSpecialMenuOS_OSCount				= 2;

			// these constants are special values that tell VW to insert the special menu item in special places...
			// When these are used as normal, they imply the same thing on both platforms.
			// You can specify different locations for different platforms by storing values in the higher bytes and setting eSpecialMenuItemInsert_UsePlatformSpecific in the lower byte.

			// For example: ESpecialMenuItemInsertLocation location = (locationMac << 8) + (locationWin << 16) + eSpecialMenuItemInsert_UsePlatformSpecific;
			// And to decode it: locationMac = ((value >> 8) & 0xFF); locationWin = ((value >> 16) 0xFF);
			// If eSpecialMenuItemInsert_UsePlatformSpecific is set, then it uses the higher bytes. If not, it uses the lower.

			// The shift value for a platform is (the above OS const + 1) * 8.. so for kSpecialMenuOS_MacOS it's 8, and kSpecialMenuOS_Windows it's 16.
			enum ESpecialMenuItemInsertLocation {
				eSpecialMenuItemInsert_First,
				eSpecialMenuItemInsert_Last,
				eSpecialMenuItemInsert_After_About,
				eSpecialMenuItemInsert_Before_About,
				eSpecialMenuItemInsert_UsePlatformSpecific = 255
			};

			// This enum describes if dividers should come before, after, or both, around a special menu item. This is using the same system as above,
			// Where the 2nd and 3rd bytes can be used to specify seperate locations per platform.
			enum ESpecialMenuItemDivider {
				eSpecialMenuItemDivider_None,
				eSpecialMenuItemDivider_BeforeItem,
				eSpecialMenuItemDivider_AfterItem,
				eSpecialMenuItemDivider_BothSides,
				eSpecialMenuItemDivider_UsePlatformSpecific = 255
			};

			/**
				Adds a menu item to be inserted at the end of workspace activation (does not modify the workspace itself).
				@param parentMenu - indicates which menu this item will be placed under, for each OS.
				@param menuItemExtensionIID - the IID of the interface for the menu to be added.
				@param insertOrder - a value used for sorting items that go in the same location.
				@param insertLocation - 
						eSpecialMenuItemInsert_First: Insert as first item
						eSpecialMenuItemInsert_Last: insert as last item
						eSpecialMenuItemInsert_After_About: insert after "About..." item (if applicable)
				@param dividers - a value defining what if any dividers to add around this menu. See ESpecialMenuItemDivider.
				@return kVCOMError_NoError if there was no error.
						There are a maximum amount of Special menu item slots. If a new one cannot be added, this will fail, with
						fLastError set to eWorkspacesError_TooManySpecialMenuItems.
				Note that adding items before the about menu item on MacOS is not possible, and will return an error.
			*/
			virtual VCOMError VCOM_CALLTYPE AddSpecialMenuItem(
				ESpecialMenuItemLocation parentMenu[kSpecialMenuOS_OSCount],
				const VWIID& menuItemExtensionIID,
				Uint32 insertOrder,
				ESpecialMenuItemInsertLocation insertLocation,
				ESpecialMenuItemDivider dividers) = 0;

			/**
				Gets the current count of Special menu items.
				@param count - the outputted count of items
				@return kVCOMError_NoError if there was no error.
			*/
			virtual VCOMError VCOM_CALLTYPE GetNumSpecialMenuItems(size_t& count) = 0;

			/**
				Gets the information about one of the Special menu items.
				@param index - the index of the requested item. If this is out of range, menuID and menuItemExtension will
					be unchanged.
				@param parentMenu - indicates which menu this item will be placed under, for each OS.
				@param menuItemExtensionIID - the IID of the interface for the menu to be added.
				@param insertOrder - a value used for sorting items that go in the same location.
				@param insertLocation - 
						eSpecialMenuItemInsert_First: Insert as first item
						eSpecialMenuItemInsert_Last: insert as last item
						eSpecialMenuItemInsert_Before_About: insert before "About..." item (if applicable)
						eSpecialMenuItemInsert_After_About: insert after "About..." item (if applicable)
						eSpecialMenuItemInsert_UsePlatformSpecific (+ ...): Use platform specific values. Byte 1 is Mac, Byte 2 is Windows.
						for example:
							Uint32 insertLocation = eSpecialMenuItemInsert_UsePlatformSpecific + 
								(((Uint32) eSpecialMenuItemInsert_Before_About) << 16) +	// Windows
								(((Uint32) eSpecialMenuItemInsert_After_About) << 8);		// Mac

						   
				@param dividers - a value defining what if any dividers to add around this menu. See ESpecialMenuItemDivider.
				@return kVCOMError_NoError if there was no error.
			*/
			virtual VCOMError VCOM_CALLTYPE GetSpecialMenuItem(
				size_t index,
				ESpecialMenuItemLocation parentMenu[kSpecialMenuOS_OSCount],
				VWIID& menuItemExtensionIID,
				Uint32& insertOrder,
				ESpecialMenuItemInsertLocation& insertLocation,
				ESpecialMenuItemDivider& dividers) = 0;

			/**
				Removes the Special menu item at requested index.
				@param index - the index of the item to remove.
				@return kVCOMError_NoError if there was no error.
			*/
			virtual VCOMError VCOM_CALLTYPE RemoveSpecialMenuItem(size_t index) = 0;

			virtual VCOMError VCOM_CALLTYPE GetProtectionFlags(const TXString& inIdentifier, Uint16& outModeFlags, Uint32& outProductFlags) = 0;

			// Delete a palette from the workspace and UI
			virtual VCOMError VCOM_CALLTYPE DeleteToolPalette(const TXString& inIdentifier) = 0;
		};

#define CREATE_IDENTIFIER_INTERNAL(itemType, resID, id, err)						\
		{																			\
			VCOMPtr<IWorkspaces> workspaces(IID_Workspaces);							\
			err = workspaces->CreateIdentifierUsingResourceID(itemType, resID, id);	\
		}

#define CREATE_IDENTIFIER_EXTERNAL(itemType, universalName, id, err)							\
		{																						\
			VCOMPtr<IWorkspaces> workspaces(IID_Workspaces);										\
			err = workspaces->CreateIdentifierUsingUniversalName(itemType, universalName, id);	\
		}
		
#define CREATE_IDENTIFIER_CONTAINER(itemType, id, err)			\
		{														\
			VCOMPtr<IWorkspaces> workspaces(IID_Workspaces);	\
			err = workspaces->CreateIdentifier(itemType, id);	\
		}
		
#define CREATE_MENUITEM_ID_INTERNAL(resID, id, err) \
			CREATE_IDENTIFIER_INTERNAL(VectorWorks::Workspaces::eWorkspaceItemType_MenuItem, resID, id, err)

#define CREATE_MENUITEM_ID_EXTERNAL(universalName, id, err) \
			CREATE_IDENTIFIER_EXTERNAL(VectorWorks::Workspaces::eWorkspaceItemType_MenuItem, universalName, id, err)

#define CREATE_MENUCHUNK_ID_INTERNAL(resID, id, err) \
			CREATE_IDENTIFIER_INTERNAL(VectorWorks::Workspaces::eWorkspaceItemType_MenuChunk, resID, id, err)

#define CREATE_MENUCHUNK_ID_EXTERNAL(universalName, id, err) \
			CREATE_IDENTIFIER_EXTERNAL(VectorWorks::Workspaces::eWorkspaceItemType_MenuChunk, universalName, id, err)

#define CREATE_MENU_ID(id, err) \
			CREATE_IDENTIFIER_CONTAINER(VectorWorks::Workspaces::eWorkspaceItemType_Menu, id, err)

#define CREATE_TOOL_ID_INTERNAL(resID, id, err) \
			CREATE_IDENTIFIER_INTERNAL(VectorWorks::Workspaces::eWorkspaceItemType_Tool, resID, id, err)

#define CREATE_TOOL_ID_EXTERNAL(universalName, id, err) \
			CREATE_IDENTIFIER_EXTERNAL(VectorWorks::Workspaces::eWorkspaceItemType_Tool, universalName, id, err)

#define CREATE_TOOLSET_ID(id, err) \
			CREATE_IDENTIFIER_CONTAINER(VectorWorks::Workspaces::eWorkspaceItemType_ToolSet, id, err)

#define CREATE_TOOLPALETTE_ID(id, err) \
			CREATE_IDENTIFIER_CONTAINER(VectorWorks::Workspaces::eWorkspaceItemType_ToolPalette, id, err)

#define CONVERT_STRING_TO_XML_SAFE(str, err)						\
		{															\
			VCOMPtr<IWorkspaces> workspaces(IID_Workspaces);		\
			err = workspaces->ConvertStringToSafeXMLFormat(str);	\
		}
		
#define CONVERT_STRING_FROM_XML_SAFE(str, err)						\
		{															\
			VCOMPtr<IWorkspaces> workspaces(IID_Workspaces);		\
			err = workspaces->ConvertSafeXMLFormatToString(str);	\
		}
		
	} // namespace Workspaces
} // namespace VectorWorks


#endif // VCOM_INTERFACE_WORKSPACES__H
